/**  @file bta_dct.c
*
*    @brief This file implements the filter (see header)
*
*    BLT_DISCLAIMER
*
*    @author Alex Falkensteiner
*
*    @cond svn
*
*    Information of last commit
*    $Rev::               $:  Revision of last commit
*    $Author::            $:  Author of last commit
*    $Date::              $:  Date of last commit
*
*    @endcond
*/


#include "bta_dct.h"
#include <stdlib.h>
#include <string.h>
#include <math.h>

#ifndef BTA_EXCLUDE_FILTERS


static uint16_t getMinUint16(uint16_t *data, uint16_t xRes, uint16_t yRes);
static uint16_t getMaxUint16(uint16_t *data, uint16_t xRes, uint16_t yRes);
static void stretchHistoUint16(uint16_t *data, uint16_t xRes, uint16_t yRes, uint16_t min, uint16_t max);
static float **malloMatFloat(int dimX, int dimY);
static void freeMat(float **m);
static void dctFloat(float **data, uint16_t xRes, uint16_t yRes, float **result);

static const float pi = 3.14159265358979323846f;


BTA_Status BFLTdctInit(BTA_FltDctConfig *config, BTA_FltHandle *handle, BTA_InfoEventInst *infoEventInst) {
    if (!handle || !config) {
        return BTA_StatusInvalidParameter;
    }
    *handle = 0;
    BTA_FltDctInst *inst = (BTA_FltDctInst *)calloc(1, sizeof(BTA_FltDctInst));
    if (!inst) {
        return BTA_StatusOutOfMemory;
    }
    inst->channelToProcess = config->channelToProcess;
    inst->channelIdResult = config->channelIdResult;
    inst->infoEventInst = infoEventInst;
    *handle = inst;
    return BTA_StatusOk;
}


BTA_Status BFLTdctClose(BTA_FltHandle *handle) {
    BTA_FltDctInst **inst = (BTA_FltDctInst **)handle;
    free(*inst);
    *inst = 0;
    return BTA_StatusOk;
}


BTA_Status BFLTdctApply(BTA_FltHandle handle, BTA_Frame **frame) {
    BTA_FltDctInst *inst = (BTA_FltDctInst *)handle;
    int chInd;
    int chCount = (*frame)->channelsLen;
    for (chInd = 0; chInd < chCount; chInd++) {
        BTA_Channel *channel = ((*frame)->channels)[chInd];
        if (channel->id == inst->channelToProcess) {
            int x, y;
            //float **outAvg = calloMatFloat(8, 8);
            //int patchCount = 0;

            switch (channel->dataFormat) {

                case BTA_DataFormatFloat32:
                    BTAinfoEventHelper(inst->infoEventInst, 3, BTA_StatusNotSupported, "BFLTdctApply: unsupported data format", channel->dataFormat);
                    return BTA_StatusNotSupported;

                case BTA_DataFormatUInt16: {
                    uint16_t *dataTemp = (uint16_t *)malloc(channel->xRes * channel->yRes * sizeof(uint16_t));
                    memcpy(dataTemp, channel->data, channel->xRes * channel->yRes * sizeof(uint16_t));
                    uint16_t min = getMinUint16(dataTemp, channel->xRes, channel->yRes);
                    uint16_t max = getMaxUint16(dataTemp, channel->xRes, channel->yRes);
                    stretchHistoUint16(dataTemp, channel->xRes, channel->yRes, min, max);


                    float **in = malloMatFloat(channel->xRes, channel->yRes);
                    float **out = malloMatFloat(channel->xRes, channel->yRes);
                    int i = 0;
                    for (y = 0; y < channel->yRes; y++) {
                        for (x = 0; x < channel->xRes; x++) {
                            in[x][y] = (float)dataTemp[i++];
                        }
                    }
                    free(dataTemp);
                    dctFloat((float **)in, channel->xRes, channel->yRes, (float **)out);
                    // we are not interested in the dc part
                    out[0][0] = 0;
                    freeMat(in);
                    i = 0;
                    float *dataNew = (float *)malloc(channel->xRes * channel->yRes * sizeof(float));
                    for (y = 0; y < channel->yRes; y++) {
                        for (x = 0; x < channel->xRes; x++) {
                            dataNew[i++] = out[x][y];
                        }
                    }
                    freeMat(out);
                    if (inst->channelIdResult) {
                        BTAinsertChannelDataIntoFrame((*frame), inst->channelIdResult, channel->xRes, channel->yRes, BTA_DataFormatFloat32, BTA_UnitUnitLess, channel->integrationTime, channel->modulationFrequency, (uint8_t *)dataNew, channel->xRes * channel->yRes * sizeof(float));
                    }
                    else {
                        BTAinfoEventHelper(inst->infoEventInst, 3, BTA_StatusNotSupported, "BFLTdctApply: channelIdResult is invalid (in place not supported)", 0);
                        return BTA_StatusNotSupported;
                    }


                    //float **in = malloMatFloat(8, 8);
                    //float **out = malloMatFloat(8, 8);
                    //int xStart, yStart;
                    //for (yStart = 0; yStart < channelTemp->yRes - 8; yStart += 4) {
                    //    for (xStart = 0; xStart < channelTemp->xRes - 8; xStart += 4) {
                    //        for (y = 0; y < 8; y++) {
                    //            for (x = 0; x < 8; x++) {
                    //                in[x][y] = (float)dataTemp[xStart + x + (yStart + y)*channelTemp->xRes];
                    //            }
                    //        }
                    ////int xi = 7, yi = 10;
                    ////for (yi = 2; yi < channelTemp->yRes / 8 - 2; yi++) {
                    ////    for (xi = 3; xi < channelTemp->xRes / 8 - 3; xi++) {
                    ////        for (y = 0; y < 8; y++) {
                    ////            for (x = 0; x < 8; x++) {
                    ////                in[x][y] = (float)dataTemp[xi * 8 + x + (yi * 8 + y)*channelTemp->xRes];
                    ////            }
                    ////        }
                    //        dctFloat((float **)in, 8, 8, (float **)out);
                    //        // we are not interested in the dc part
                    //        //out[0][0] = 0;
                    //        for (y = 0; y < 8; y++) {
                    //            for (x = 0; x < 8; x++) {
                    //                outAvg[x][y] += (float)fabs(out[x][y]);
                    //                if (outAvg[x][y] < 0) {
                    //                    outAvg[x][y] = 3.402823466e+38F;
                    //                }
                    //            }
                    //        }
                    //        patchCount++;
                    //    }
                    //}
                    //freeMat(in);
                    //freeMat(out);
                    break;
                }

                default:
                    BTAinfoEventHelper(inst->infoEventInst, 3, BTA_StatusNotSupported, "BFLTdctApply: unsupported data format", channel->dataFormat);
                    return BTA_StatusNotSupported;
            }

            //for (y = 0; y < 8; y++) {
            //    for (x = 0; x < 8; x++) {
            //        outAvg[x][y] = outAvg[x][y] / patchCount;// (channelTemp->xRes - 64)*(channelTemp->yRes - 24) / 64;
            //    }
            //}

            //// Calculate the sum of / diagonals
            //float outDiagAvg[8] = { 0 };
            //for (x = 0; x < 8; x++) {
            //    float nSum = 0;
            //    int i;
            //    for (i = 0; i <= x; i++) {
            //        nSum += outAvg[x - i][i];
            //    }
            //    outDiagAvg[x] = nSum / i;
            //}
            //freeMat(outAvg);

            //inst->result0 = outDiagAvg[0];
            //inst->result1 = outDiagAvg[1];
            //inst->result2 = outDiagAvg[2];
            //inst->result3 = outDiagAvg[3];
            //inst->result4 = outDiagAvg[4];
            //inst->result5 = outDiagAvg[5];
            //inst->result6 = outDiagAvg[6];
            //inst->result7 = outDiagAvg[7];

            return BTA_StatusOk;
        }
    }
    return BTA_StatusOk;
}



static uint16_t getMinUint16 (uint16_t *data, uint16_t xRes, uint16_t yRes) {
    int i = 0;
    uint16_t min = 0xffff;
    uint16_t *dataTemp = data;
    for (i = 0; i < xRes * yRes; i++) {
        if (*dataTemp < min) {
            min = *dataTemp;
        }
        dataTemp++;
    }
    return min;
}

static uint16_t getMaxUint16(uint16_t *data, uint16_t xRes, uint16_t yRes) {
    int i = 0;
    uint16_t max = 0;
    uint16_t *dataTemp = data;
    for (i = 0; i < xRes * yRes; i++) {
        if (*dataTemp > max) {
            max = *dataTemp;
        }
        dataTemp++;
    }
    return max;
}


static void stretchHistoUint16(uint16_t *data, uint16_t xRes, uint16_t yRes, uint16_t min, uint16_t max) {
    int i = 0;
    uint16_t *dataTemp = data;
    float factor = (float)0xffff / (float)(max - min);
    for (i = 0; i < xRes * yRes; i++) {
        *dataTemp = (unsigned short)((float)(*dataTemp - min) * factor);
        dataTemp++;
    }
}


static float **malloMatFloat(int dimX, int dimY) {
    float **m = (float **)malloc(dimX * sizeof(float*));
    float *p = (float *)malloc(dimX*dimY * sizeof(float));
    int i;
    for (i = 0; i <dimX; i++) {
        m[i] = &p[i*dimY];
    }
    return m;
}


static void freeMat(float **m) {
    free(m[0]);
    free(m);
}


static void dctFloat(float **data, uint16_t xRes, uint16_t yRes, float **result) {
    uint16_t x2, y2, x, y;
    for (x = 0; x < xRes; ++x) {
        for (y = 0; y < yRes; ++y) {
            result[x][y] = 0;
            for (x2 = 0; x2 < xRes; x2++) {
                for (y2 = 0; y2 < yRes; y2++) {
                    result[x][y] += (float)(data[x2][y2] * cos(pi / ((float)xRes)*(x2 + 1. / 2.)*x)*cos(pi / ((float)yRes)*(y2 + 1. / 2.)*y));
                }
            }
        }
    }
}

#endif
